<?php


namespace RTMTrade\OpenCart\AdminApi\Mapping;


use AutoMapperPlus\AutoMapperInterface;
use AutoMapperPlus\Configuration\AutoMapperConfigInterface;
use AutoMapperPlus\MappingOperation\Operation;
use RTMTrade\OpenCart\AdminApi\Model\Keyword;
use RTMTrade\OpenCart\AdminApi\Model\MetaDescription;
use RTMTrade\OpenCart\AdminApi\Model\Product;
use RTMTrade\OpenCart\AdminApi\Model\ProductAttribute;
use RTMTrade\OpenCart\AdminApi\Model\ProductAttributeDescription;
use RTMTrade\OpenCart\AdminApi\Model\ProductCategory;
use RTMTrade\OpenCart\AdminApi\Model\ProductDescription;
use RTMTrade\OpenCart\AdminApi\Model\ProductDiscount;
use RTMTrade\OpenCart\AdminApi\Model\ProductOptions;
use RTMTrade\OpenCart\AdminApi\Model\ProductOptionValue;
use RTMTrade\OpenCart\AdminApi\Model\ProductSpecial;
use RTMTrade\OpenCart\AdminApi\Model\Review;
use stdClass;

class ProductMapper implements ObjectMapperInterface
{

    public static function registerMappings(AutoMapperConfigInterface $config)
    {
        self::registerMetaDescription($config);
        self::registerReview($config);
        self::registerProductDescription($config);
        self::registerProductAttributeDescription($config);
        self::registerProductDiscount($config);
        self::registerProductSpecial($config);
        self::registerProductOptionValue($config);
        self::registerProductOptions($config);
        self::registerProduct($config);
    }

    protected static function registerMetaDescription(AutoMapperConfigInterface $config): void
    {
        $config->registerMapping(stdClass::class, MetaDescription::class)
            // ignore id for common meta description
            ->forMember('id', Operation::mapFrom(fn ($source) => property_exists($source, 'id') ? $source->id : null));
    }

    protected static function registerReview(AutoMapperConfigInterface $config): void
    {
        $config->registerMapping(stdClass::class, Review::class)
            // parse datetime
            ->forMember('dateAdded',
                Operation::mapFrom(ObjectMapper::dateMapper('d/m/Y H:i:s', 'date_added')));
    }

    protected static function registerProductDescription(AutoMapperConfigInterface $config): void
    {
        $config->registerMapping(stdClass::class, ProductDescription::class);
    }

    protected static function registerProductAttributeDescription(AutoMapperConfigInterface $config): void
    {
        $config->registerMapping(stdClass::class, ProductAttributeDescription::class);
    }

    protected static function registerProductDiscount(AutoMapperConfigInterface $config): void
    {
        $config->registerMapping(stdClass::class, ProductDiscount::class)
            // format dateStart
            ->forMember('dateStart',
                Operation::mapFrom(ObjectMapper::dateMapper('Y-m-d', 'date_start')))
            // format dateEnd
            ->forMember('dateEnd',
                Operation::mapFrom(ObjectMapper::dateMapper('Y-m-d', 'date_end')));
    }

    protected static function registerProductSpecial(AutoMapperConfigInterface $config): void
    {
        $config->registerMapping(stdClass::class, ProductSpecial::class)
            // format dateStart
            ->forMember('dateStart',
                Operation::mapFrom(ObjectMapper::dateMapper('Y-m-d', 'date_start')))
            // format dateEnd
            ->forMember('dateEnd',
                Operation::mapFrom(ObjectMapper::dateMapper('Y-m-d', 'date_end')));
    }

    protected static function registerProductOptionValue(AutoMapperConfigInterface $config): void
    {
        $config->registerMapping(stdClass::class, ProductOptionValue::class)
            // map id
            ->forMember('id',
                Operation::fromProperty('product_option_value_id'))
            // map misspelled field
            ->forMember('priceFormatted',
                Operation::fromProperty('price_formated'));
    }

    protected static function registerProductOptions(AutoMapperConfigInterface $config): void
    {
        $config->registerMapping(stdClass::class, ProductOptions::class)
            // format dateEnd
            ->forMember('id',
                Operation::fromProperty('product_option_id'))
            // map renamed field
            ->forMember('values',
                Operation::mapFrom(fn(object $source, AutoMapperInterface $mapper) => $mapper->mapMultiple($source->option_value, ProductOptionValue::class))
            );
    }

    protected static function registerProduct(AutoMapperConfigInterface $config): void
    {
        $config->registerMapping(stdClass::class, Product::class)
            ->forMember('manufacturerId',
                Operation::mapFrom(fn ($source) => empty($source->manufacturer_id) ?: intval($source->manufacturer_id)))
            // map misspelled field
            ->forMember('priceFormatted',
                Operation::fromProperty('price_formated'))
            // map renamed field
            ->forMember('relatedProductIds',
                Operation::mapFrom(fn(object $source) => array_map('intval', $source->product_relateds)))
            // map specific formatted object
            ->forMember('reviews',
                Operation::mapFrom(function (object $source, AutoMapperInterface $mapper) {
                    $collection = [];
                    if (isset($source->reviews) && ($source->reviews->review_total ?? 0) > 0) {
                        foreach ($source->reviews->reviews ?? [] as $review) {
                            $collection[] = $mapper->map($review, Review::class);
                        }
                    }
                    return $collection;
                }))
            // map collection
            ->forMember('productDescriptions',
                ObjectMapper::mapMultipleFromProperty(ProductDescription::class, 'product_description'))
            // map specific formatted object
            ->forMember('productAttributes',
                Operation::mapFrom(function (object $source, AutoMapperInterface $mapper) {
                    $attributes = $source->product_attributes;
                    $collection = [];

                    foreach ($attributes as $id => $attribute) {
                        $descriptions = [];
                        foreach ($attribute as $idx => $attr) {
                            /** @var ProductAttributeDescription $description */
                            $description = $mapper->map($attr, ProductAttributeDescription::class);
                            $description->setId($idx);

                            $descriptions[] = $description;
                        }
                        $collection[] = new ProductAttribute(intval($id), $descriptions);
                    }

                    return $collection;
                }))
            // map collection
            ->forMember('discounts',
                Operation::mapCollectionTo(ProductDiscount::class))
            // map collection
            ->forMember('special',
                Operation::mapCollectionTo(ProductSpecial::class))
            // map collection
            ->forMember('options',
                Operation::mapCollectionTo(ProductOptions::class))
            // map specific formatted object
            ->forMember('categories',
                Operation::mapFrom(self::productCategoryMapping()))
            // map currency fields
            ->forMember('currency',
                Operation::mapFrom(ObjectMapper::currencyMapper()))
            ->forMember('dateAvailable',
                Operation::mapFrom(ObjectMapper::dateMapper('Y-m-d', 'date_available')))
            ->forMember('dateAdded',
                Operation::mapFrom(ObjectMapper::dateMapper('Y-m-d H:i:s', 'date_added')))
            ->forMember('dateModified',
                Operation::mapFrom(ObjectMapper::dateMapper('Y-m-d H:i:s', 'date_modified')))
            ->forMember('reward',
                Operation::mapFrom(fn($source) => intval($source->reward)))
            ->forMember('keyword',
                Operation::mapFrom(function ($source) {
                    $keywords = $source->keyword;
                    $out = [];
                    foreach ($keywords as $keywordLanguages) {
                        foreach ((array)$keywordLanguages as $languageId => $keyword) {
                            $out[] = new Keyword($languageId, $keyword);
                        }
                    }
                    return $out;
                })
            )
        ;
    }

    protected static function productCategoryMapping(): \Closure
    {
        return function (object $source, AutoMapperInterface $mapper) {
            $collection = [];
            foreach ($source->category as $catTranslations) {
                $category = new ProductCategory();
                $category->setId($catTranslations[0]->category_id);

                foreach ($catTranslations as $cat) {
                    /** @var MetaDescription $description */
                    $description = $mapper->map($cat, MetaDescription::class);
                    $description->setId($cat->category_id);
                    $category->addDescription($description);
                }
                $collection[] = $category;
            }
            return $collection;
        };
    }
}