<?php

namespace RTMTrade\OpenCart\AdminApi\Api;

use AutoMapperPlus\AutoMapperInterface;
use AutoMapperPlus\Exception\UnregisteredMappingException;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;
use Psr\Log\LoggerInterface;
use RTMTrade\OpenCart\AdminApi\ApiException;
use RTMTrade\OpenCart\AdminApi\Configuration;
use RTMTrade\OpenCart\AdminApi\HeaderSelector;
use RTMTrade\OpenCart\AdminApi\Model\Store;
use RTMTrade\OpenCart\AdminApi\Model\StoreStats;

class StoreApi extends AbstractApi
{

    public function __construct(ClientInterface $client, Configuration $config, HeaderSelector $headerSelector, AutoMapperInterface $mapper, LoggerInterface $logger = null)
    {
        parent::__construct($client, $config, $headerSelector, $mapper, $logger);
        $this->resourceName = 'stores';
        $this->resourceClass = Store::class;

        $this->allowedOperations = [
            'get',
            'update',
            'list',
            'addSubresourceById' => [
                'copy' => Store::class
            ],
            'listBySubresource' => [
                'stats' => StoreStats::class
            ]
        ];
    }

    /**
     * @param int $id the store ID to copy
     * @param Store $store
     * @return int the new store ID
     * @throws ApiException
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function copyShop(int $id, Store $store): int
    {
        [$response] = $this->copyShopWithHttpInfo($id, $store);
        return $response->getData();
    }

    /**
     * @param int $id
     * @param Store $store
     * @return array
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     * @throws ApiException
     */
    public function copyShopWithHttpInfo(int $id, Store $store): array
    {
        $request = $this->createAddSubresourceByIdRequest($id, 'copy', $store);

        return $this->sendRequest($request);
    }

    public function getStats()
    {
        [$response] = $this->getStatsWithHttpInfo();
        return $this->mapper->map($response->getData(), StoreStats::class);
    }

    public function getStatsWithHttpInfo(): array
    {
        $request = $this->createGetBySubresourceRequest('stats');
        return $this->sendRequest($request);
    }

}
