<?php


namespace RTMTrade\OpenCart\AdminApi\Api;


use AutoMapperPlus\AutoMapperInterface;
use AutoMapperPlus\Exception\UnregisteredMappingException;
use GuzzleHttp\Client;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\Psr7\Request;
use GuzzleHttp\RequestOptions;
use Psr\Http\Message\ResponseInterface;
use RTMTrade\OpenCart\AdminApi\ApiException;
use RTMTrade\OpenCart\AdminApi\Configuration;
use RTMTrade\OpenCart\AdminApi\HeaderSelector;
use RTMTrade\OpenCart\AdminApi\Model\ApiResponse;
use RuntimeException;

abstract class BaseHttpApi
{

    protected ClientInterface $client;
    protected Configuration $config;
    protected HeaderSelector $headerSelector;
    protected AutoMapperInterface $mapper;

    public function __construct(
        ClientInterface $client,
        Configuration $config,
        HeaderSelector $headerSelector,
        AutoMapperInterface $mapper
    )
    {
        $this->client = $client;
        $this->config = $config;
        $this->headerSelector = $headerSelector;
        $this->mapper = $mapper;
    }

    /**
     * @return ClientInterface
     */
    public function getClient(): ClientInterface
    {
        return $this->client;
    }

    /**
     * @param ClientInterface $client
     * @return BaseHttpApi
     */
    public function setClient(ClientInterface $client): BaseHttpApi
    {
        $this->client = $client;
        return $this;
    }

    /**
     * @return Configuration
     */
    public function getConfig(): Configuration
    {
        return $this->config;
    }

    /**
     * @param Configuration $config
     * @return BaseHttpApi
     */
    public function setConfig(Configuration $config): BaseHttpApi
    {
        $this->config = $config;
        return $this;
    }

    /**
     * @return HeaderSelector
     */
    public function getHeaderSelector(): HeaderSelector
    {
        return $this->headerSelector;
    }

    /**
     * @param HeaderSelector $headerSelector
     * @return BaseHttpApi
     */
    public function setHeaderSelector(HeaderSelector $headerSelector): BaseHttpApi
    {
        $this->headerSelector = $headerSelector;
        return $this;
    }

    /**
     * @return AutoMapperInterface
     */
    public function getMapper(): AutoMapperInterface
    {
        return $this->mapper;
    }

    /**
     * @param AutoMapperInterface $mapper
     * @return BaseHttpApi
     */
    public function setMapper(AutoMapperInterface $mapper): BaseHttpApi
    {
        $this->mapper = $mapper;
        return $this;
    }

    /**
     * Create http client option
     *
     * @return array of http client options
     * @throws RuntimeException on file opening failure
     */
    protected function createHttpClientOption(): array
    {
        $options = [];
        if ($this->config->getDebug()) {
            $options[RequestOptions::DEBUG] = fopen($this->config->getDebugFile(), 'a');
            if (!$options[RequestOptions::DEBUG]) {
                throw new RuntimeException('Failed to open the debug file: ' . $this->config->getDebugFile());
            }
        }

        return $options;
    }

    /**
     * Add authorization and user-agent headers to array.
     * @param array $headers
     * @return array
     */
    protected function addDefaultHeaders(array $headers): array
    {
        // add auth header
        $apiKey = $this->config->getApiKeyWithPrefix('Authorization');
        if ($apiKey !== null) {
            $headers['Authorization'] = $apiKey;
        }

        // add user-agent header
        if ($this->config->getUserAgent()) {
            $headers['User-Agent'] = $this->config->getUserAgent();
        }
        return $headers;
    }

    /**
     * @param Request $request
     * @return array
     * @throws ApiException
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    protected function sendRequest(Request $request): array
    {
        try {
            $options = $this->createHttpClientOption();
            try {
                $response = $this->client->send($request, $options);
            } catch (RequestException $e) {
                throw new ApiException(
                    "[{$e->getCode()}] {$e->getMessage()}",
                    $e->getCode(),
                    $e->getResponse() ? $e->getResponse()->getHeaders() : null,
                    $e->getResponse() ? (string)$e->getResponse()->getBody() : null,
                    $e
                );
            }

            return $this->asApiResponse($response);
        } catch (ApiException $e) {
            switch ($e->getCode()) {
                case 200:
                    $e->setResponseObject($this->mapper->map(
                        $e->getResponseBody(),
                        ApiResponse::class
                    ));
                    break;
            }
            throw $e;
        }
    }

    /**
     * @param ResponseInterface $response
     * @return array
     * @throws ApiException
     * @throws UnregisteredMappingException
     */
    protected function asApiResponse(ResponseInterface $response): array
    {
        $statusCode = $response->getStatusCode();

        if ($statusCode < 200 || $statusCode > 299) {
            // throw exception if response was not in OK range
            throw new ApiException(
                "[{$statusCode}] Error connecting to the Opencart API",
                $statusCode,
                $response->getHeaders(),
                $response->getBody()
            );
        }

        // deserialize
        $json = \GuzzleHttp\json_decode((string)$response->getBody());

        return [
            $this->mapper->map($json, ApiResponse::class),
            $response->getStatusCode(),
            $response->getHeaders()
        ];
    }


}