<?php


namespace RTMTrade\OpenCart\AdminApi\Api;

use AutoMapperPlus\AutoMapperInterface;
use AutoMapperPlus\Exception\UnregisteredMappingException;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Psr7\Request;
use InvalidArgumentException;
use Psr\Log\LoggerInterface;
use RTMTrade\OpenCart\AdminApi\ApiException;
use RTMTrade\OpenCart\AdminApi\Configuration;
use RTMTrade\OpenCart\AdminApi\HeaderSelector;
use RTMTrade\OpenCart\AdminApi\Model\ApiResponse;
use RTMTrade\OpenCart\AdminApi\Model\Category;
use SplFileObject;

/**
 * Class CategoryApi
 * @package RTMTrade\OpenCart\AdminApi\Api
 * @extends AbstractApi<Category>
 */
class CategoryApi extends AbstractApi
{

    public function __construct(ClientInterface $client, Configuration $config, HeaderSelector $headerSelector, AutoMapperInterface $mapper, LoggerInterface $logger = null)
    {
        parent::__construct($client, $config, $headerSelector, $mapper, $logger);
        $this->resourceClass = Category::class;
        $this->resourceName = 'categories';

        // setup operations extendable from AbstractApi
        $this->allowedOperations = [
            'get',
            'add',
            'update',
            'delete',
            'list',
            'bulkDelete',
            'paginationSubresource' => [
                'extended' => Category::class // extended pagination
            ],
            'listBySubresource' => [
                'level' => Category::class, // listByLevel
                'parent' => Category::class // listByParentId
            ]
        ];


    }

    /**
     * Get list of categories by depth level
     *
     * @param int $level Level number (required)
     *
     * @return Category[]
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function listByLevel(int $level): array
    {
        [$response] = $this->listByLevelWithHttpInfo($level);
        return $this->mapper->mapMultiple($response->getData(), Category::class);
    }

    /**
     * Get list of categories by depth level
     *
     * @param int $level Level number (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function listByLevelWithHttpInfo(int $level): array
    {
        $request = $this->createListBySubresourceRequest('level', $level);
        return $this->sendRequest($request);
    }

    /**
     * Get list of categories by parent ID
     *
     * @param int $parentId Parent Category ID (required)
     *
     * @return Category[]
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function listByParentId(int $parentId): array
    {
        [$response] = $this->listByParentIdWithHttpInfo($parentId);

        return $this->mapper->mapMultiple($response->getData(), Category::class);
    }

    /**
     * Get list of categories by parent ID
     *
     * @param int $parentId Parent Category ID (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function listByParentIdWithHttpInfo(int $parentId): array
    {
        $request = $this->createListBySubresourceRequest('parent', $parentId);

        return $this->sendRequest($request);
    }


    /**
     * Get list of categories by parent ID and level
     *
     * @param int $level Level (required)
     * @param int $parent Parent Category ID (required)
     *
     * @return Category[]
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function listByParentIdAndLevel(int $level, int $parent): array
    {
        [$response] = $this->listByParentIdAndLevelWithHttpInfo($level, $parent);
        return $this->mapper->mapMultiple($response->getData(), Category::class);
    }

    /**
     * Get list of categories by parent ID and level
     *
     * @param int $level Level (required)
     * @param int $parent Parent Category ID (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function listByParentIdAndLevelWithHttpInfo(int $level, int $parent): array
    {
        $request = $this->createCategoriesByParentIdAndLevelRequest($level, $parent);
        return $this->sendRequest($request);
    }

    /**
     * Create request for operation 'byParentId1'
     *
     * @param int $level Level (required)
     * @param int $parent Parent Category ID (required)
     *
     * @return Request
     * @throws InvalidArgumentException
     */
    protected function createCategoriesByParentIdAndLevelRequest(int $level, int $parent): Request
    {
        $headers = $this->headerSelector->selectHeaders(
            ['application/json'],
            []
        );

        $headers = $this->addDefaultHeaders($headers);
        $uri = "{$this->config->getBaseUrl()}/categories/parent/{$parent}/level/{$level}";
        return new Request('GET', $uri, $headers);
    }


    /**
     * Get list of categories (all in one request)
     *
     * @param int $limit Limit (required)
     * @param int $page Page (required)
     *
     * @return ApiResponse
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function extendedListPagination(int $limit, int $page): ApiResponse
    {
        [$response] = $this->extendedListPaginationWithHttpInfo($limit, $page);
        return $response;
    }

    /**
     * Get list of categories (all in one request)
     *
     * @param int $limit Limit (required)
     * @param int $page Page (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function extendedListPaginationWithHttpInfo(int $limit, int $page): array
    {
        $request = $this->createPaginationSubresourceRequest('extended', $limit, $page);

        return $this->sendRequest($request);
    }

    /**
     * Add image to category by category ID
     *
     * @param int $id Category Id (required)
     * @param SplFileObject $file (required)
     *
     * @return ApiResponse
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function updateImage(int $id, SplFileObject $file): ApiResponse
    {
        [$response] = $this->updateImageWithHttpInfo($id, $file);
        return $response;
    }

    /**
     * Add image to category by category ID
     *
     * @param int $id Category Id (required)
     * @param SplFileObject $file (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function updateImageWithHttpInfo(int $id, SplFileObject $file): array
    {
        $request = $this->createPostFileRequest("categories/{$id}", $file);

        return $this->sendRequest($request);
    }
}
