<?php


namespace RTMTrade\OpenCart\AdminApi;


use AutoMapperPlus\AutoMapper;
use AutoMapperPlus\Exception\UnregisteredMappingException;
use DateTime;
use Exception;
use GuzzleHttp\Client;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;
use Psr\Log\LoggerInterface;
use RTMTrade\OpenCart\AdminApi\Api\CategoryApi;
use RTMTrade\OpenCart\AdminApi\Api\CouponApi;
use RTMTrade\OpenCart\AdminApi\Api\CustomerApi;
use RTMTrade\OpenCart\AdminApi\Api\CustomerGroupApi;
use RTMTrade\OpenCart\AdminApi\Api\CustomFieldsApi;
use RTMTrade\OpenCart\AdminApi\Api\ManufacturerApi;
use RTMTrade\OpenCart\AdminApi\Api\OrderApi;
use RTMTrade\OpenCart\AdminApi\Api\PaymentMethodApi;
use RTMTrade\OpenCart\AdminApi\Api\ProductAttributeApi;
use RTMTrade\OpenCart\AdminApi\Api\ProductAttributeGroupsApi;
use RTMTrade\OpenCart\AdminApi\Api\ProductFeaturedApi;
use RTMTrade\OpenCart\AdminApi\Api\ProductFilterApi;
use RTMTrade\OpenCart\AdminApi\Api\ProductLatestApi;
use RTMTrade\OpenCart\AdminApi\Api\ProductOptionApi;
use RTMTrade\OpenCart\AdminApi\Api\ProductOptionValueApi;
use RTMTrade\OpenCart\AdminApi\Api\ProductsApi;
use RTMTrade\OpenCart\AdminApi\Api\SearchApi;
use RTMTrade\OpenCart\AdminApi\Api\ShippingMethodApi;
use RTMTrade\OpenCart\AdminApi\Api\StoreApi;
use RTMTrade\OpenCart\AdminApi\Api\SystemDataApi;
use RTMTrade\OpenCart\AdminApi\Api\TokenApi;
use RTMTrade\OpenCart\AdminApi\Api\UserApi;
use RTMTrade\OpenCart\AdminApi\Api\VoucherApi;
use RTMTrade\OpenCart\AdminApi\Mapping\ObjectMapper;
use RTMTrade\OpenCart\AdminApi\Model\Login;

class OpenCartAdminApi
{

    /**
     * @var Configuration
     */
    private Configuration $configuration;

    /**
     * @var ClientInterface
     */
    private $client;
    /**
     * @var HeaderSelector
     */
    private HeaderSelector $headerSelector;

    /**
     * @var string
     */
    private string $clientId;
    /**
     * @var string
     */
    private string $clientSecret;


    private ?Token $token;
    private ?LoggerInterface $logger;
    private AutoMapper $objectMapper;

    public function __construct(ClientInterface $client = null,
                                Configuration $configuration = null,
                                HeaderSelector $headerSelector = null,
                                LoggerInterface $logger = null)
    {
        $this->client = $client ?: new Client();
        $this->configuration = $configuration ?: new Configuration();
        $this->headerSelector = $headerSelector ?: new HeaderSelector();
        $this->token = null;
        $this->logger = $logger;

        $this->objectMapper = new AutoMapper(ObjectMapper::init());
    }

    /**
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     * @throws ApiException
     */
    public function resetAuthorization()
    {
        ($this->token && !$this->token->isExpired()) && $this->users()->logout();
    }

    /**
     * @return AutoMapper
     */
    public function getMapper(): AutoMapper
    {
        return $this->objectMapper;
    }


    /**
     * @return bool
     * @throws ApiException
     * @throws OpencartUserAlreadyLoggedException
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function authorize()
    {
        if (is_null($this->token) || $this->token->isExpired()) {
            $this->logger->debug('Token is null or expired, getting new token.');
            $tokenApi = new TokenApi($this->client, $this->configuration, $this->headerSelector, $this->objectMapper);
            [$response, $code, $headers] = $tokenApi->getTokenWithHttpInfo(
                'Basic ' . base64_encode("{$this->clientId}:{$this->clientSecret}"),
                'client_credentials');

            if (!is_null($response) && empty($response->getError())) {
                $data = $response->getData();

                $this->token = $this->objectMapper->map($data, Token::class);
                $this->token->setSince(new DateTime());

                $this->logger->debug('Got a new token, logging in user using Authorization ' . $this->configuration->getApiKeyWithPrefix('Authorization'));

                $this->setAuthorizationConfig($this->token);

                return $this->loginUser();
            } else {
                throw new ApiException("Error fetching access token.", $code, $headers);
            }
        } elseif (!$this->token->isExpired()) {
            $this->setAuthorizationConfig($this->token);

            try {
                return $this->loginUser();
            } catch (ApiException $throwable) {
                if ($throwable->getCode() === 400) {
                    $body = json_decode($throwable->getResponseBody(), true);

                    if (!$body) {
                        throw new Exception("Could not parse json from API endpoint.", 400, $throwable);
                    }

                    // Opencart Admin API response if user is already logged in.
                    // todo: this should not happen since loginUser() was implemented; delete?
                    if (($body['error'][0] ?? '') === 'User already is logged') {
                        throw new OpencartUserAlreadyLoggedException("User already logged in", 400, $throwable);
                        // todo: return true?
                    }
                }
                throw $throwable;
            }
        }
        return false;
    }

    /**
     * @param string $fileName
     * @throws UnregisteredMappingException should never happen
     */
    public function readToken(string $fileName)
    {
        // create temp folder if not exist
        if (!is_dir($this->configuration->getTempFolderPath() . '/')) {
            mkdir($this->configuration->getTempFolderPath() . '/', 0777, true);
        }

        $filePath = $this->configuration->getTempFolderPath() . '/' . $fileName;

        // check if file exists
        if (is_file($filePath)
            && (($size = filesize($filePath)) > 0)
            && ($file = fopen($filePath, 'r'))) {
            // read contents
            $fileContents = fread($file, $size);

            // if reading file succeeded
            if ($fileContents !== false) {
                // decode json contents
                $contents = json_decode($fileContents);
                if (isset($contents->access_token)
                    && isset($contents->expires_in)
                    && isset($contents->token_type)
                    && isset($contents->since)) {
                    // fill token object
                    $this->objectMapper->map($contents, Token::class);
                }
            }
        }
    }

    public function writeToken(string $file)
    {
        if (!is_dir($this->configuration->getTempFolderPath() . '/')) {
            mkdir($this->configuration->getTempFolderPath() . '/', 0777, true);
        }

        $filePath = $this->configuration->getTempFolderPath() . '/' . $file;
        $file = fopen($filePath, 'wb');
        if (!is_null($this->token)) {
            $string = json_encode([
                'access_token' => $this->token->getAccessToken(),
                'expires_in' => $this->token->getExpiresIn(),
                'token_type' => $this->token->getTokenType(),
                'since' => $this->token->getSince()->format('Y-m-d H:i:s')
            ]);
            fwrite($file, $string);
        }
    }

    public function resetToken(string $file): bool
    {
        if (!is_dir($this->configuration->getTempFolderPath() . '/')) {
            // temp dir doesn't exist, no token present
            return true;
        }

        // delete file if exists
        $filePath = $this->configuration->getTempFolderPath() . '/' . $file;
        return is_file($filePath) && unlink($filePath);
    }

    /**
     * @return bool
     * @throws ApiException
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    protected function loginUser()
    {
        try {
            // is user already logged in?
            $this->users()->getCurrentUserDetails();
            return true;
        } catch (ApiException $throwable) {
            // user is not logged in, do login request
            [$response] = $this->users()->loginWithHttpInfo(new Login(
                $this->configuration->getUsername(),
                $this->configuration->getPassword()
            ));

            return $response->getSuccess();
        }
    }

    /**
     * @param string $clientId
     */
    public function setClientId(string $clientId)
    {
        $this->clientId = $clientId;
    }

    /**
     * @param string $clientSecret
     */
    public function setClientSecret(string $clientSecret)
    {
        $this->clientSecret = $clientSecret;
    }

    /**
     * @return ClientInterface
     */
    public function getClient()
    {
        return $this->client;
    }

    /**
     * @return string
     */
    public function getClientId(): string
    {
        return $this->clientId;
    }

    /**
     * @return string
     */
    public function getClientSecret(): string
    {
        return $this->clientSecret;
    }

    /**
     * @return Configuration
     */
    public function getConfiguration(): Configuration
    {
        return $this->configuration;
    }

    /**
     * @return HeaderSelector
     */
    public function getHeaderSelector(): HeaderSelector
    {
        return $this->headerSelector;
    }

    /**
     * @return Token
     */
    public function getToken(): ?Token
    {
        return $this->token;
    }

    /* APIs */

    public function categories(): CategoryApi
    {
        return new CategoryApi($this->client, $this->configuration, $this->headerSelector, $this->objectMapper);
    }

    public function coupons(): CouponApi
    {
        return new CouponApi($this->client, $this->configuration, $this->headerSelector, $this->objectMapper);
    }

    public function customers(): CustomerApi
    {
        return new CustomerApi($this->client, $this->configuration, $this->headerSelector, $this->objectMapper);
    }

    public function customerGroups(): CustomerGroupApi
    {
        return new CustomerGroupApi($this->client, $this->configuration, $this->headerSelector, $this->objectMapper);
    }

    public function customFields(): CustomFieldsApi
    {
        return new CustomFieldsApi($this->client, $this->configuration, $this->headerSelector, $this->objectMapper);
    }

    public function manufacturers(): ManufacturerApi
    {
        return new ManufacturerApi($this->client, $this->configuration, $this->headerSelector, $this->objectMapper);
    }

    public function orders(): OrderApi
    {
        return new OrderApi($this->client, $this->configuration, $this->headerSelector, $this->objectMapper);
    }

    public function paymentMethods(): PaymentMethodApi
    {
        return new PaymentMethodApi($this->client, $this->configuration, $this->headerSelector, $this->objectMapper);
    }

    public function productAttributes(): ProductAttributeApi
    {
        return new ProductAttributeApi($this->client, $this->configuration, $this->headerSelector, $this->objectMapper);
    }

    public function productAttributeGroups(): ProductAttributeGroupsApi
    {
        return new ProductAttributeGroupsApi($this->client, $this->configuration, $this->headerSelector, $this->objectMapper);
    }

    public function productFeatured(): ProductFeaturedApi
    {
        return new ProductFeaturedApi($this->client, $this->configuration, $this->headerSelector, $this->objectMapper);
    }

    public function productFilter(): ProductFilterApi
    {
        return new ProductFilterApi($this->client, $this->configuration, $this->headerSelector, $this->objectMapper);
    }

    public function productLatest(): ProductLatestApi
    {
        return new ProductLatestApi($this->client, $this->configuration, $this->headerSelector, $this->objectMapper);
    }

    public function productOptions(): ProductOptionApi
    {
        return new ProductOptionApi($this->client, $this->configuration, $this->headerSelector, $this->objectMapper);
    }

    public function productOptionValues(): ProductOptionValueApi
    {
        return new ProductOptionValueApi($this->client, $this->configuration, $this->headerSelector, $this->objectMapper);
    }

    public function products(): ProductsApi
    {
        return new ProductsApi($this->client,
            $this->configuration,
            $this->headerSelector,
            $this->objectMapper,
            $this->logger);
    }

    public function shippingMethods(): ShippingMethodApi
    {
        return new ShippingMethodApi($this->client, $this->configuration, $this->headerSelector, $this->objectMapper);
    }

    public function stores(): StoreApi
    {
        return new StoreApi($this->client, $this->configuration, $this->headerSelector, $this->objectMapper);
    }

    public function systemData(): SystemDataApi
    {
        return new SystemDataApi($this->client, $this->configuration, $this->headerSelector, $this->objectMapper);
    }

    public function users(): UserApi
    {
        return new UserApi($this->client, $this->configuration, $this->headerSelector, $this->objectMapper);
    }

    public function vouchers(): VoucherApi
    {
        return new VoucherApi($this->client, $this->configuration, $this->headerSelector, $this->objectMapper);
    }

    public function search(): SearchApi
    {
        return new SearchApi($this->client, $this->configuration, $this->headerSelector, $this->objectMapper);
    }

    protected function setAuthorizationConfig(Token $token): void
    {
        $this->configuration = $this->configuration->setAccessToken($token->getAccessToken());

        $this->configuration->setApiKeyPrefix("Authorization", $token->getTokenType());
        $this->configuration->setApiKey("Authorization", $token->getAccessToken());
    }
}