<?php


namespace RTMTrade\OpenCart\AdminApi\Mapping;


use AutoMapperPlus\AutoMapperInterface;
use AutoMapperPlus\Configuration\AutoMapperConfig;
use AutoMapperPlus\Configuration\AutoMapperConfigInterface;
use AutoMapperPlus\MappingOperation\Operation;
use AutoMapperPlus\NameConverter\NamingConvention\CamelCaseNamingConvention;
use AutoMapperPlus\NameConverter\NamingConvention\SnakeCaseNamingConvention;
use Closure;
use DateTime;
use RTMTrade\OpenCart\AdminApi\Model\Address;
use RTMTrade\OpenCart\AdminApi\Model\ApiResponse;
use RTMTrade\OpenCart\AdminApi\Model\Currency;
use RTMTrade\OpenCart\AdminApi\Model\OrderStatus;
use RTMTrade\OpenCart\AdminApi\Model\SimpleCurrency;
use RTMTrade\OpenCart\AdminApi\Model\Customer;
use RTMTrade\OpenCart\AdminApi\Model\CustomField;
use RTMTrade\OpenCart\AdminApi\Model\Language;
use RTMTrade\OpenCart\AdminApi\Model\Login;
use RTMTrade\OpenCart\AdminApi\Model\StockStatus;
use RTMTrade\OpenCart\AdminApi\Model\User;
use RTMTrade\OpenCart\AdminApi\Token;
use stdClass;

class ObjectMapper
{

    public static function init(): AutoMapperConfigInterface
    {
        $config = new AutoMapperConfig();
        $config->getOptions()->setSourceMemberNamingConvention(new SnakeCaseNamingConvention());
        $config->getOptions()->setDestinationMemberNamingConvention(new CamelCaseNamingConvention());
        $config->getOptions()->setShouldSkipConstructor(false);

        // common mappings
        $config->registerMapping(stdClass::class, ApiResponse::class);

        $config->registerMapping(stdClass::class, Address::class)
            // map id
            // map id
            ->forMember('id',
                Operation::fromProperty('address_id'))
            // todo: these might not be needed but im lazy and better safe than sorry
            ->forMember('address1',
                Operation::fromProperty('address_1'))
            ->forMember('address2',
                Operation::fromProperty('address_2'))
            ->forMember('isoCode2',
                Operation::fromProperty('iso_code_2'))
            ->forMember('isoCode3',
                Operation::fromProperty('iso_code_3'));

        $config->registerMapping(stdClass::class, SimpleCurrency::class)
            ->forMember('id', Operation::fromProperty('currency_id'))
            ->forMember('code', Operation::fromProperty('currency_code'))
            ->forMember('value', Operation::fromProperty('currency_value'));

        $config->registerMapping(stdClass::class, StockStatus::class)
            ->forMember('id', Operation::fromProperty('stock_status_id'));
        $config->registerMapping(stdClass::class, OrderStatus::class)
            ->forMember('id', Operation::fromProperty('order_status_id'));

        $config->registerMapping(stdClass::class, Language::class)
            ->forMember('id', Operation::fromProperty('language_id'));

        $config->registerMapping(stdClass::class, Currency::class)
            ->forMember('id', Operation::fromProperty('currency_id'))
            ->forMember('dateModified', Operation::mapFrom(self::dateMapper('Y-m-d H:i:s', 'date_modified')))
        ;

        self::initUserMappings($config);

        CustomerMapper::registerMappings($config);
        ProductMapper::registerMappings($config);
        OrderMapper::registerMappings($config);

        return $config;
    }

    private static function initUserMappings(AutoMapperConfig $config)
    {
        $config->registerMapping(stdClass::class, Token::class)
            ->forMember('since',
                Operation::mapFrom(self::dateMapper('Y-m-d H:i:s', 'since')))
            ->reverseMap();

        $config->registerMapping(stdClass::class, Login::class)
            ->reverseMap();

        $config->registerMapping(stdClass::class, User::class)
            ->forMember('id',
                Operation::fromProperty('user_id'))
            ->forMember('dateAdded',
                Operation::mapFrom(ObjectMapper::dateMapper('Y-m-d H:i:s', 'date_added')))
            ->reverseMap()
            // other way around
            ->forMember('user_id',
                Operation::fromProperty('id'))
            ->forMember('date_added',
                Operation::mapFrom(ObjectMapper::dateToString('Y-m-d H:i:s', 'dateAdded')));

    }

    private static function initCustomerMapping(AutoMapperConfig $config)
    {
        $config->registerMapping(stdClass::class, CustomField::class)
            ->forMember('id',
                Operation::fromProperty('custom_field_id'));

        $config->registerMapping(stdClass::class, Customer::class)
            // map customer_id to $id
            ->forMember('id',
                Operation::fromProperty('customer_id'))
            // parse date_added
            ->forMember('dateAdded',
                Operation::mapFrom(ObjectMapper::dateMapper('d/m/Y', 'date_added')))
            // map custom fields collection
            ->forMember('customFields',
                Operation::mapCollectionTo(CustomField::class))
            // map addresses collection
            ->forMember('addresses',
                Operation::mapCollectionTo(Address::class));
    }

    private static function initOrderMapping(AutoMapperConfig $config)
    {
    }

    public static function currencyMapper(): Closure
    {
        return fn(object $source) => new SimpleCurrency($source->currency_id ?? null, $source->currency_code ?? null, $source->currency_value ?? null);
    }

    /**
     * Use to remap shipping and payment addresses fields to common address fields.
     * @param object $source
     * @param string $prefix
     * @return object
     */
    public static function remapAddressNames(object $source, string $prefix): object
    {
        return (object)[
            'firstname' => $source->{"{$prefix}_firstname"},
            'lastname' => $source->{"{$prefix}_lastname"},
            'company' => $source->{"{$prefix}_company"},
            'address_1' => $source->{"{$prefix}_address_1"},
            'address_2' => $source->{"{$prefix}_address_2"},
            'postcode' => $source->{"{$prefix}_postcode"},
            'city' => $source->{"{$prefix}_city"},
            'zone_id' => $source->{"{$prefix}_zone_id"},
            'zone' => $source->{"{$prefix}_zone"},
            'zone_code' => $source->{"{$prefix}_zone_code"},
            'country_id' => $source->{"{$prefix}_country_id"},
            'country' => $source->{"{$prefix}_country"},
            'iso_code_2' => $source->{"{$prefix}_iso_code_2"},
            'iso_code_3' => $source->{"{$prefix}_iso_code_3"},
            'address_format' => $source->{"{$prefix}_address_format"},
            'custom_field' => $source->{"{$prefix}_custom_field"},
        ];
    }

    public static function remapMethodDescriptorArray(object $source, string $prefix): object
    {
        return (object)[
            'title' => $source->{"{$prefix}_method"},
            'code' => $source->{"{$prefix}_code"},
        ];
    }

    public static function dateMapper(string $format, string $field): Closure
    {
        return function (object $source) use ($format, $field) {
            $getter = 'get' . ucfirst($field);
            if (method_exists($source, $getter)) {
                return DateTime::createFromFormat($format, $source->{$getter}());
            }
            if (property_exists($source, $field) && !empty($source->{$field})) {
                return DateTime::createFromFormat($format, $source->{$field});
            } else {
                return null;
            }
        };
    }

    public static function dateToString(string $format, string $field)
    {
        return fn(object $source) => $source->{'get' . ucfirst($field)}->format($format);
    }

    /**
     * Map an array of objects to array of ids with key '$name'.
     * @param array $objects
     * @param string $name
     * @return array of ids with key $name
     */
    public static function asDeleteObject(array $objects, string $name)
    {
        return [$name => array_map(fn($o) => $o->getId(), $objects)];
    }

    public static function mapMultipleFromProperty(string $class, string $field)
    {
        return Operation::mapFrom(function (object $source, AutoMapperInterface $mapper) use ($class, $field) {
            $getter = 'get' . ucfirst($field);
            if (method_exists($source, $getter)) {
                return $mapper->mapMultiple($source->{$getter}(), $class);
            }
            return $mapper->mapMultiple($source->{$field}, $class);
        });
    }

    /**
     * @param string $class
     * @return Closure
     */
    public static function flatMap(string $class): Closure
    {
        return fn(object $source, AutoMapperInterface $mapper) => $mapper->map($source, $class);
    }

}