<?php

namespace RTMTrade\OpenCart\AdminApi\Api;

use AutoMapperPlus\AutoMapperInterface;
use AutoMapperPlus\Exception\UnregisteredMappingException;
use DateTime;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Psr7\Request;
use InvalidArgumentException;
use Psr\Log\LoggerInterface;
use RTMTrade\OpenCart\AdminApi\ApiException;
use RTMTrade\OpenCart\AdminApi\Configuration;
use RTMTrade\OpenCart\AdminApi\HeaderSelector;
use RTMTrade\OpenCart\AdminApi\Model\ApiResponse;
use RTMTrade\OpenCart\AdminApi\Model\Customer;
use RTMTrade\OpenCart\AdminApi\Model\Reward;
use RTMTrade\OpenCart\AdminApi\Model\Transaction;
use RTMTrade\OpenCart\AdminApi\ObjectSerializer;
use stdClass;

class CustomerApi extends AbstractApi
{

    public function __construct(ClientInterface $client, Configuration $config, HeaderSelector $headerSelector, AutoMapperInterface $mapper, LoggerInterface $logger = null)
    {
        parent::__construct($client, $config, $headerSelector, $mapper, $logger);
        $this->resourceClass = Customer::class;
        $this->resourceName = 'customers';

        $this->allowedOperations = [
            'get',
            'add',
            'update',
            'delete',
            'pagination',
            'bulkDelete',
            'getBySubresource' => [
                'email' => Customer::class,
            ],
            'listBySubresource' => [
                'added_from' => Customer::class,
                'added_to' => Customer::class,
            ]
        ];
    }

    /**
     * Get customer details by email
     *
     * @param string $email Customer email (required)
     *
     * @return ApiResponse
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function getByEmail(string $email): ApiResponse
    {
        [$response] = $this->getByEmailWithHttpInfo($email);
        return $response;
    }

    /**
     * Get customer details by email
     *
     * @param string $email Customer email (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function getByEmailWithHttpInfo(string $email): array
    {
        $request = $this->createGetBySubresourceRequest('email', $email);
        return $this->sendRequest($request);
    }

    /**
     * Get list of customers filtered by date parameter (between)
     *
     * @param DateTime $addedFrom From date added (required)
     * @param DateTime $addedTo To date added (required)
     *
     * @return ApiResponse
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function listAddedBetween(DateTime $addedFrom, DateTime $addedTo): ApiResponse
    {
        [$response] = $this->listAddedBetweenWithHttpInfo($addedFrom, $addedTo);
        return $response;
    }

    /**
     * Get list of customers filtered by date parameter (between)
     *
     * @param DateTime $addedFrom From date added (required)
     * @param DateTime $addedTo To date added (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function listAddedBetweenWithHttpInfo(DateTime $addedFrom, DateTime $addedTo): array
    {
        $request = $this->createListAddedBetweenRequest($addedFrom, $addedTo);

        return $this->sendRequest($request);
    }

    /**
     * Create request for operation 'customerListAddedBetween'
     *
     * @param DateTime $addedFrom From date added (required)
     * @param DateTime $addedTo To date added (required)
     *
     * @return Request
     * @throws InvalidArgumentException
     */
    protected function createListAddedBetweenRequest(DateTime $addedFrom, DateTime $addedTo): Request
    {
        $headers = $this->headerSelector->selectHeaders(
            ['application/json'],
            []
        );
        $headers = $this->addDefaultHeaders($headers);

        $uri = $this->config->getBaseUrl() . sprintf("/customers/added_from/%s/added_to/%s",
                ObjectSerializer::toPathValue($addedFrom),
                ObjectSerializer::toPathValue($addedTo));
        return new Request('GET', $uri, $headers);
    }

    /**
     * Get list of customers filtered by date parameter (from)
     *
     * @param DateTime $addedFrom From date added (required)
     *
     * @return ApiResponse
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function listAddedFrom(DateTime $addedFrom): ApiResponse
    {
        [$response] = $this->listAddedFromWithHttpInfo($addedFrom);
        return $response;
    }

    /**
     * Get list of customers filtered by date parameter (from)
     *
     * @param DateTime $addedFrom From date added (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function listAddedFromWithHttpInfo(DateTime $addedFrom): array
    {
        $request = $this->createListBySubresourceRequest('added_from', $addedFrom);

        return $this->sendRequest($request);
    }

    /**
     * Get list of customers filtered by  date parameter (on)
     *
     * @param DateTime $addedOn Customer added on date (required)
     *
     * @return ApiResponse
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function listAddedOn(DateTime $addedOn): ApiResponse
    {
        [$response] = $this->listAddedOnWithHttpInfo($addedOn);
        return $response;
    }

    /**
     * Get list of customers filtered by  date parameter (on)
     *
     * @param DateTime $addedOn Customer added on date (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function listAddedOnWithHttpInfo(DateTime $addedOn): array
    {
        $request = $this->createListBySubresourceRequest('added_on', $addedOn);

        return $this->sendRequest($request);
    }

    /**
     * Add reward points to customer
     *
     * @param int $id Customer Id (required)
     * @param Reward $rewardObject Reward object (required)
     *
     * @return ApiResponse
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function rewardAdd(int $id, Reward $rewardObject): ApiResponse
    {
        [$response] = $this->rewardAddWithHttpInfo($id, $rewardObject);
        return $response;
    }

    /**
     * Add reward points to customer
     *
     * @param int $id Customer Id (required)
     * @param Reward $rewardObject Reward object (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function rewardAddWithHttpInfo(int $id, Reward $rewardObject): array
    {
        $request = $this->createRewardAddRequest($id, $rewardObject);

        return $this->sendRequest($request);
    }

    /**
     * Create request for operation 'rewardAdd'
     *
     * @param int $id Customer Id (required)
     * @param Reward $rewardObject Reward object (required)
     *
     * @return Request
     * @throws UnregisteredMappingException
     */
    protected function createRewardAddRequest(int $id, Reward $rewardObject): Request
    {
        $headers = $this->headerSelector->selectHeaders(
            ['application/json'],
            ['application/json']
        );

        $httpBody = \GuzzleHttp\json_encode($this->mapper->map($rewardObject, stdClass::class));

        $headers = $this->addDefaultHeaders($headers);

        $uri = "{$this->config->getBaseUrl()}/reward/{$id}";
        return new Request('POST', $uri, $headers, $httpBody);
    }

    /**
     * Add transaction points to customer
     *
     * @param int $id Customer Id (required)
     * @param Transaction $transactionObject Transaction object (required)
     *
     * @return ApiResponse
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function transactionAdd(int $id, Transaction $transactionObject): ApiResponse
    {
        [$response] = $this->transactionAddWithHttpInfo($id, $transactionObject);
        return $response;
    }

    /**
     * Add transaction points to customer
     *
     * @param int $id Customer Id (required)
     * @param Transaction $transactionObject Transaction object (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function transactionAddWithHttpInfo(int $id, Transaction $transactionObject): array
    {
        $request = $this->createTransactionAddRequest($id, $transactionObject);

        return $this->sendRequest($request);
    }

    /**
     * Create request for operation 'transactionAdd'
     *
     * @param int $id Customer Id (required)
     * @param Transaction $transactionObject Transaction object (required)
     *
     * @return Request
     * @throws UnregisteredMappingException
     */
    protected function createTransactionAddRequest(int $id, Transaction $transactionObject): Request
    {
        $headers = $this->headerSelector->selectHeaders(
            ['application/json'],
            ['application/json']
        );

        $httpBody = \GuzzleHttp\json_encode($this->mapper->map($transactionObject, stdClass::class));

        $headers = $this->addDefaultHeaders($headers);

        $uri = "{$this->config->getBaseUrl()}/transaction/{$id}";
        return new Request('POST', $uri, $headers, $httpBody);
    }
}
