<?php


namespace RTMTrade\OpenCart\AdminApi;


use DateTime;
use GuzzleHttp\Client;
use GuzzleHttp\ClientInterface;
use RTMTrade\OpenCart\AdminApi\Api\CategoryApi;
use RTMTrade\OpenCart\AdminApi\Api\CouponApi;
use RTMTrade\OpenCart\AdminApi\Api\CustomerApi;
use RTMTrade\OpenCart\AdminApi\Api\CustomerGroupApi;
use RTMTrade\OpenCart\AdminApi\Api\CustomFieldsApi;
use RTMTrade\OpenCart\AdminApi\Api\ManufacturerApi;
use RTMTrade\OpenCart\AdminApi\Api\OrderApi;
use RTMTrade\OpenCart\AdminApi\Api\OrderCreateOrderApi;
use RTMTrade\OpenCart\AdminApi\Api\PaymentMethodApi;
use RTMTrade\OpenCart\AdminApi\Api\ProductAttributeApi;
use RTMTrade\OpenCart\AdminApi\Api\ProductAttributeGroupsApi;
use RTMTrade\OpenCart\AdminApi\Api\ProductFeaturedApi;
use RTMTrade\OpenCart\AdminApi\Api\ProductFilterApi;
use RTMTrade\OpenCart\AdminApi\Api\ProductLatestApi;
use RTMTrade\OpenCart\AdminApi\Api\ProductOptionApi;
use RTMTrade\OpenCart\AdminApi\Api\ProductOptionValueApi;
use RTMTrade\OpenCart\AdminApi\Api\ProductsApi;
use RTMTrade\OpenCart\AdminApi\Api\ShippingMethodsApi;
use RTMTrade\OpenCart\AdminApi\Api\StoreApi;
use RTMTrade\OpenCart\AdminApi\Api\SystemDataApi;
use RTMTrade\OpenCart\AdminApi\Api\TokenApi;
use RTMTrade\OpenCart\AdminApi\Api\UserApi;
use RTMTrade\OpenCart\AdminApi\Api\VoucherApi;
use RTMTrade\OpenCart\AdminApi\Model\Login;

class OpenCartAdminApi
{

    /**
     * @var Configuration
     */
    private $configuration;

    /**
     * @var ClientInterface
     */
    private $client;
    /**
     * @var HeaderSelector
     */
    private $headerSelector;

    /**
     * @var string
     */
    private $clientId;
    /**
     * @var string
     */
    private $clientSecret;

    /**
     * @var Token
     */
    private $token;

    public function __construct(ClientInterface $client = null,
                                Configuration $configuration = null,
                                HeaderSelector $headerSelector = null)
    {
        $this->client = $client ? $client : new Client();
        $this->configuration = $configuration ? $configuration : new Configuration();
        $this->headerSelector = $headerSelector ? $headerSelector : new HeaderSelector();
        $this->token = null;
    }

    public function authorize()
    {
        if (is_null($this->token) || $this->token->isExpired()) {
            $tokenApi = new TokenApi($this->client, $this->configuration, $this->headerSelector);
            list($response, $code, $headers) = $tokenApi->tokenGetWithHttpInfo('Basic ' . base64_encode("{$this->clientId}:{$this->clientSecret}"), 'client_credentials');

            if (!is_null($response) && empty($response->getError())) {
                $data = $response->getData();

                $this->token = (new Token())
                    ->setAccessToken($data['access_token'])
                    ->setTokenType($data['token_type'])
                    ->setExpiresIn($data['expires_in'])
                    ->setSince(new DateTime());

                // todo: save token to file in cache

                $this->configuration = $this->configuration->setAccessToken($this->token->getAccessToken());

                $this->configuration->setApiKeyPrefix("Authorization", $this->token->getTokenType());
                $this->configuration->setApiKey("Authorization", $this->token->getAccessToken());

                // login
                $loginResponse = $this->users()->loginUser(new Login([
                    'username' => $this->configuration->getUsername(),
                    'password' => $this->configuration->getPassword()
                ]));

                return $loginResponse->valid();
            } else {
                throw new ApiException("Error fetching access token.", $code, $headers);
            }
        } elseif (!$this->token->isExpired()) {
            $this->configuration = $this->configuration->setAccessToken($this->token->getAccessToken());

            $this->configuration->setApiKeyPrefix("Authorization", $this->token->getTokenType());
            $this->configuration->setApiKey("Authorization", $this->token->getAccessToken());

            try {
                // login
                $loginResponse = $this->users()->loginUser(new Login([
                    'username' => $this->configuration->getUsername(),
                    'password' => $this->configuration->getPassword()
                ]));
            } catch (ApiException $throwable) {
                if ($throwable->getCode() === 400) {
                    // user already logged in
                    return true;
                }
            }

            return $loginResponse->valid();
        }
        return false;
    }

    public function readToken(string $fileName)
    {
        if (!is_dir($this->configuration->getTempFolderPath() . '/')) {
            mkdir($this->configuration->getTempFolderPath() . '/', 0777, true);
        }

        $filePath = $this->configuration->getTempFolderPath() . '/' . $fileName;

        if (is_file($filePath)) {
            // open filestream
            $file = fopen($filePath, 'r');

            // make sure the file has content
            if (($size = filesize($filePath)) > 0) {
                // read contents
                $fileContents = fread($file, $size);

                // if reading file succeeded
                if ($fileContents !== false) {
                    // decode json contents
                    $contents = json_decode($fileContents, true);
                    if (isset($contents['accessToken']) && isset($contents['expiresIn'])
                        && isset($contents['tokenType']) && isset($contents['since'])) {
                        // fill token object
                        $this->token = new Token();
                        $this->token->setAccessToken($contents['accessToken']);
                        $this->token->setExpiresIn($contents['expiresIn']);
                        $this->token->setTokenType($contents['tokenType']);
                        $this->token->setSince((new DateTime())->setTimestamp($contents['since']));
                    }
                }
            }
        }
    }

    public function writeToken(string $file)
    {
        if (!is_dir($this->configuration->getTempFolderPath() . '/')) {
            mkdir($this->configuration->getTempFolderPath() . '/', 0777, true);
        }

        $filePath = $this->configuration->getTempFolderPath() . '/' . $file;
        $file = fopen($filePath, 'wb');
        if (!is_null($this->token)) {
            $string = json_encode([
                'accessToken' => $this->token->getAccessToken(),
                'expiresIn' => $this->token->getExpiresIn(),
                'tokenType' => $this->token->getTokenType(),
                'since' => $this->token->getSince()->getTimestamp()
            ]);
            fwrite($file, $string);
        }
    }

    /**
     * @param string $clientId
     */
    public function setClientId($clientId)
    {
        $this->clientId = $clientId;
    }

    /**
     * @param string $clientSecret
     */
    public function setClientSecret($clientSecret)
    {
        $this->clientSecret = $clientSecret;
    }

    /**
     * @return ClientInterface
     */
    public function getClient()
    {
        return $this->client;
    }

    /**
     * @return string
     */
    public function getClientId()
    {
        return $this->clientId;
    }

    /**
     * @return string
     */
    public function getClientSecret()
    {
        return $this->clientSecret;
    }

    /**
     * @return Configuration
     */
    public function getConfiguration()
    {
        return $this->configuration;
    }

    /**
     * @return HeaderSelector
     */
    public function getHeaderSelector()
    {
        return $this->headerSelector;
    }

    /**
     * @return Token
     */
    public function getToken()
    {
        return $this->token;
    }

    /* APIs */

    public function categories()
    {
        return new CategoryApi($this->client, $this->configuration, $this->headerSelector);
    }

    public function coupons()
    {
        return new CouponApi($this->client, $this->configuration, $this->headerSelector);
    }

    public function customers()
    {
        return new CustomerApi($this->client, $this->configuration, $this->headerSelector);
    }

    public function customerGroups()
    {
        return new CustomerGroupApi($this->client, $this->configuration, $this->headerSelector);
    }

    public function customFields()
    {
        return new CustomFieldsApi($this->client, $this->configuration, $this->headerSelector);
    }

    public function manufacturers()
    {
        return new ManufacturerApi($this->client, $this->configuration, $this->headerSelector);
    }

    public function orders()
    {
        return new OrderApi($this->client, $this->configuration, $this->headerSelector);
    }

    public function createOrders()
    {
        return new OrderCreateOrderApi($this->client, $this->configuration, $this->headerSelector);
    }

    public function paymentMethods()
    {
        return new PaymentMethodApi($this->client, $this->configuration, $this->headerSelector);
    }

    public function productAttributes()
    {
        return new ProductAttributeApi($this->client, $this->configuration, $this->headerSelector);
    }

    public function productAttributeGroups()
    {
        return new ProductAttributeGroupsApi($this->client, $this->configuration, $this->headerSelector);
    }

    public function productFeatured()
    {
        return new ProductFeaturedApi($this->client, $this->configuration, $this->headerSelector);
    }

    public function productFilter()
    {
        return new ProductFilterApi($this->client, $this->configuration, $this->headerSelector);
    }

    public function productLatest()
    {
        return new ProductLatestApi($this->client, $this->configuration, $this->headerSelector);
    }

    public function productOptions()
    {
        return new ProductOptionApi($this->client, $this->configuration, $this->headerSelector);
    }

    public function productOptionValues()
    {
        return new ProductOptionValueApi($this->client, $this->configuration, $this->headerSelector);
    }

    public function products()
    {
        return new ProductsApi($this->client, $this->configuration, $this->headerSelector);
    }

    public function shippingMethods()
    {
        return new ShippingMethodsApi($this->client, $this->configuration, $this->headerSelector);
    }

    public function stores()
    {
        return new StoreApi($this->client, $this->configuration, $this->headerSelector);
    }

    public function systemData()
    {
        return new SystemDataApi($this->client, $this->configuration, $this->headerSelector);
    }

    public function users()
    {
        return new UserApi($this->client, $this->configuration, $this->headerSelector);
    }

    public function vouchers()
    {
        return new VoucherApi($this->client, $this->configuration, $this->headerSelector);
    }
}