<?php

namespace pCon;

defined( 'ABSPATH' ) || exit;

class_exists( 'pCon\pConLoader' ) || exit;

/**
 * Register all actions and filters for the plugin
 */
class pConLoader {


	/**
	 * The array of actions registered with WordPress
	 *
	 * @var array
	 */
	protected array $actions;

	/**
	 * The array of filters registered with WordPress
	 *
	 * @var array
	 */
	protected array $filters;

	/**
	 * The array of shortcodes registered with WordPress
	 *
	 * @var array
	 */
	protected array $shortcodes;

	/**
	 * Initialize the collections used to maintain the actions and filters
	 */
	public function __construct() {
		$this->actions = array();
		$this->filters = array();
		$this->shortcodes = array();
	}

	/**
	 * Add a new action to the collection to be registered with WordPress
	 *
	 * @param string $hook WordPress hook name.
	 * @param object $component reference object.
	 * @param string $callback callback method.
	 * @param int $priority execution priority.
	 * @param int $accepted_args amount of accepted arguments.
	 *
	 * @return void
	 */
	public function add_action( string $hook, object $component, string $callback, int $priority = 10, int $accepted_args = 1 ): void {
		$this->actions = $this->add( $this->actions, $hook, $component, $callback, $priority, $accepted_args );
	}

	/**
	 * Add a new filter to the collection to be registered with WordPress
	 *
	 * @param string $hook WordPress hook name.
	 * @param object $component reference object.
	 * @param string $callback callback method.
	 * @param int $priority execution priority.
	 * @param int $accepted_args amount of accepted arguments.
	 *
	 * @return void
	 */
	public function add_filter( string $hook, object $component, string $callback, int $priority = 10, int $accepted_args = 1 ): void {
		$this->filters = $this->add( $this->filters, $hook, $component, $callback, $priority, $accepted_args );
	}

	/**
	 * Add a new shortcodes to the collection to be registered with WordPress
	 *
	 * @param string $hook WordPress hook name.
	 * @param object $component reference object.
	 * @param string $callback callback method.
	 *
	 * @return void
	 */
	public function add_shortcode( string $hook, object $component, string $callback ): void {
		$this->shortcodes = $this->add( $this->shortcodes, $hook, $component, $callback, 10, 1 );
	}

	/**
	 * A utility function that is used to register the actions and hooks into a single
	 *
	 * @param array $hooks all registered hooks.
	 * @param string $hook WordPress hook name.
	 * @param object $component reference object.
	 * @param string $callback callback method.
	 * @param int $priority execution priority.
	 * @param int $accepted_args amount of accepted arguments.
	 *
	 * @return array
	 */
	private function add( array $hooks, string $hook, object $component, string $callback, int $priority, int $accepted_args ): array {
		$hooks[] = array(
			'hook'          => $hook,
			'component'     => $component,
			'callback'      => $callback,
			'priority'      => $priority,
			'accepted_args' => $accepted_args,
		);

		return $hooks;
	}

	/**
	 * Register the filters and actions with WordPress
	 *
	 * @return void
	 */
	public function run(): void {
		foreach ( $this->filters as $hook ) {
			add_filter(
				$hook['hook'],
				array(
					$hook['component'],
					$hook['callback'],
				),
				$hook['priority'],
				$hook['accepted_args']
			);
		}

		foreach ( $this->actions as $hook ) {
			add_action(
				$hook['hook'],
				array(
					$hook['component'],
					$hook['callback'],
				),
				$hook['priority'],
				$hook['accepted_args']
			);
		}

		foreach ( $this->shortcodes as $hook ) {
			add_shortcode(
				$hook['hook'],
				array(
					$hook['component'],
					$hook['callback'],
				),
			);
		}
	}
}
